module net.BurtonRadons.dedit.highlight.python;

import net.BurtonRadons.dedit.main;
import std.path;
import std.string;
import std.ctype;

alias std.ctype.isdigit isdigit;

/** Python highlighter. */
class Python_Highlighter : SyntaxHighlighter
{
    char [] [] reservedBase = 
    [
        "and",
        "break",
        "class",
        "continue",
        "def",
        "elif",
        "else",
        "except",
        "finally",
        "for",
        "from",
        "if",
        "import",
        "in",
        "is",
        "not",
        "or",
        "pass",
        "print",
        "raise",
        "return",
        "try",
        "while",
    ];

    bit [char []] reserved;

    const char [] symbols = "()[]<>{}:;=!%^&*-+|/.~,$";

    bit [char []] keywordDict (char [] [] base)
    {
        bit [char []] dict;

        for (int c; c < base.length; c ++)
            dict [base [c]] = true;

        return dict;
    }

    this ()
    {
        reserved = keywordDict (reservedBase);
    }

    static this ()
    {
        list ~= new Python_Highlighter ();
    }

    override char [] name () { return "Python"; }
    override char [] exts () { return "*.py"; }

    override float match (char [] filename, char [] [] data)
    {
        char [] ext = std.string.tolower (std.path.getExt (filename));

        if (ext == "py")
            return 1;
        return 0;
    }

    final bit isSymbol (char f)
    {
        for (int c; c < symbols.length; c ++)
            if (f == symbols [c])
                return true;

        return false;
    }

    final bit isIdentifierStart (char f) { return isalpha (f) || f == '_'; }
    final bit isIdentifierMiddle (char f) { return isalnum (f) || f == '_'; }

    final char [] getKeyword (char *c, int n)
    {
        int d;

        if (n == 0 || !isIdentifierStart (*c))
            return null;
        for (d = 1; d < n; d ++)
            if (!isIdentifierMiddle (c [d]))
                break;

        return c [0 .. d];
    }
    
    struct LineInfo
    {
        char code; /**< Current syntax highlighting code. */
        char open;
            /**< Current major type:
               * <ul>
               * <li>'*' - multiline comment.
               * <li>'"' - double-quoted std.string.
               * <li>"'" - single-quoted std.string.
               * <li>'i' - identifier.
               * <li>'/' - single-line comment.
               * <li>'#' - number.
               * </ul>
               */
    }

    override int extraSize () { return LineInfo.size; }

    override void highlight (char [] line, char [] high, void *lastp, void *nextp)
    {
        LineInfo *last = (LineInfo *) lastp;
        LineInfo *next = (LineInfo *) nextp;
        char *c, h, e;
        char open, code;

        if (last !== null)
        {
            open = last.open;
            code = last.code;
        }

        c = line;
        h = high;
        e = c + line.length;

        while (c < e)
        {
            int n = (int) (e - c);
            char f = *c;
            char [] r;

        restart:
            if (open == '"')
            {
                if (c [0] == '\\')
                {
                    *h ++ = code;
                    c ++;
                    if (c < e)
                        goto def;
                    goto dun;
                }
                else if (c [0] == '"')
                {
                    *h ++ = code;
                    code = open = 0;
                    c ++;
                }
                else
                    goto def;
            }
            else if (open == '\'')
            {
                if (c [0] == '\\')
                {
                    *h ++ = code;
                    c ++;
                    if (c < e)
                        goto def;
                    goto dun;
                }
                else if (c [0] == '\'')
                {
                    *h ++ = code;
                    code = open = 0;
                    c ++;
                }
                else
                    goto def;
            }
            else if (open == '/')
                goto def;
            else if (open == '#')
            {
                if (isdigit (f) || f == 'x' || f == 'X' || f == '.' || f == 'e' || f == 'E'
                 || f == 'a' || f == 'b' || f == 'c' || f == 'd' || f == 'e' || f == 'f'
                 || f == 'A' || f == 'B' || f == 'C' || f == 'D' || f == 'E' || f == 'F'
                 || f == 'l' || f == 'L')
                    goto def;
                else
                {
                    code = open = 0;
                    goto restart;
                }
            }
        /* open == 0 from here on */
            else if (f == '\"')
            {
                open = code = '"';
                goto def;
            }
            else if (f == '\'')
            {
                open = '\''; 
                code = '"';
                goto def;
            }
            else if (f == '#')
            {
                open = '/';
                code = '*';
                goto def;
            }
            else if (isdigit (f) || (f == '.' && (n == 1 || isdigit (c [1]))))
            {
                open = code = '#';
                goto def;
            }
            else
            {
                if (isSymbol (f))
                {
                    *h ++ = 's';
                    c ++;
                }
                else if ((r = getKeyword (c, n)) !== null)
                {
                    if (r == "self" || r == "None")
                        h [0 .. r.length] = "m";
                    else
                        h [0 .. r.length] = (r in reserved) ? 'r' : 'i';
                    c += r.length;
                    h += r.length;
                }
                else
                    goto def;
            }

        dun:
            continue;
        def:
            *h ++ = code;
            c += 1;
        }

        if (open == 'i' || open == '/' || open == '#')
            open = 0;

        next.code = code;
        next.open = open;
    }
}
